'''
Created on 26/12/2013

@author: David Vilares
'''
from nltk.parse.dependencygraph import DependencyGraph

class SentimentDependencyGraphNodeKeys(object):
    """
    The keys of a node of a L{SentimentDependencyGraph}
    """

    SEMANTIC_ORIENTATION = "SEMANTIC_ORIENTATION"
    INTENSIFICATION = "INTENSIFICATION"
    SUBJECTIVITY = "SUBJECTIVITY"
    POSITIVE_WORDS = "POSITIVE_WORDS"
    NEGATIVE_WORDS = "NEGATIVE_WORDS"
    NUMBER_OF_INTENSIFIERS = "NUMBER_OF_INTENSIFIERS"
    LENGTH_TEXT = "LENGTH_TEXT"
    NUMBER_OF_WORDS = "NUMBER_OF_WORDS"  
    LINGUISTIC_INFO = "LINGUISTIC_INFO"  
    WORD = "word"
    REL = "rel"
    DEPS = "deps"
    TAG = "tag"
    ADDRESS = "address"
    HEAD = "head"
    
class SentimentJSONKeys(object):
    """
    The keys used to transform a L{SentimentDependencyGraph} to JSON
    """
    CHILDREN = "children"
    SEMANTIC_ORIENTATION = "so"
    IS_NEGATION = "is_negation"
    IS_INTENSIFIER = "is_intensifier"
    POSTAG = "postag"
    WORD = "word"
    ADDRESS = "address"
    DEPENDENCY_TYPE = "dependency_type"

class SentimentDependencyGraph(DependencyGraph):
    '''
    An extension of the class L{nltk.parse.dependencygraph.DependencyGraph}
    to include sentiment information
    '''
    ROOT_WORD = "ROOT_WORD"
    


    def __init__(self, tree_str=None):
        '''
        Constructor
        @param tree_str: See L{nltk.parse.dependencygraph.DependencyGraph}
        '''
        super(SentimentDependencyGraph,self).__init__(tree_str)


    def contains_sentiment_info(self,address):
        try:
            self.get_by_address(address)[SentimentDependencyGraphNodeKeys.SEMANTIC_ORIENTATION]
            return True
        except:
            return False
        
    def set_linguistic_info(self,node,linguistic_info):
        node[SentimentDependencyGraphNodeKeys.LINGUISTIC_INFO] = linguistic_info


    def get_linguistic_info(self,node):
        return node[SentimentDependencyGraphNodeKeys.LINGUISTIC_INFO]
    
    def get_rel(self,node):
        """
        @param node: A node of a L{nltk.parse.dependencygraph.DependencyGraph}
        @return: dependency relation with head's node
        """
        return node[SentimentDependencyGraphNodeKeys.REL] 
    

    def get_deps(self,node):
        """
        @param node: A node of a L{nltk.parse.dependencygraph.DependencyGraph}
        @return: A list of children id's of node
        """
        return node[SentimentDependencyGraphNodeKeys.DEPS]
    
    
    def _get_adversative_id(self,node):
        """
        It obtains the adversative node identifier.
        @precondition: The parameter tag must follow the regexp for adversative tags: tag:additional_information@id
        @param tag: The tag of the artificial adversative node
        @return: An integer 
        """
        return int(node[SentimentDependencyGraphNodeKeys.TAG].split(":")[1].split("@")[1])
    
    def get_tag(self,node):
        """
        @param node: A node of a L{nltk.parse.dependencygraph.DependencyGraph}
        @return: The fine PoS-tag of  the node
        """
        return node[SentimentDependencyGraphNodeKeys.TAG].split('@')[0]
    
    
    def get_ctag(self,node):
        """
        @param node: A node of a L{nltk.parse.dependencygraph.DependencyGraph}
        @return: The coarse PoS-tag of the node
        """
        return node[SentimentDependencyGraphNodeKeys.TAG].split(':')[0]
     

    def get_head(self,node):
        return node[SentimentDependencyGraphNodeKeys.HEAD]

    def get_address(self,node):
        """
        @param node: A node of a L{nltk.parse.dependencygraph.DependencyGraph}
        @return: The position in the sentence of the node. Zero is the root node
        """
        return node[SentimentDependencyGraphNodeKeys.ADDRESS]


    def get_word(self,node):
        """
        @return: The word of the node
        """
        if self.is_root_node(node):
            return self.ROOT_WORD
        else:
            return node[SentimentDependencyGraphNodeKeys.WORD]
      
    def get_lexical_category(self,node):
        """
        @param node:A node of a L{nltk.parse.dependencygraph.DependencyGraph}
        @return: The lexical category of the node
        """
        return self.get_tag(node).split(':')[0]

    def is_leaf(self,node): 
        """
        @param node: A node of a DependencyGraph
        @return: True if is a leaf node, False otherwise 
        """     
        return self.get_deps(node) == [] 
    
    def is_root_node(self,node):
        """
        @param node: A node of a L{nltk.parse.dependencygraph.DependencyGraph}
        @return: True if is the root of the dependency graph, False otherwise
        """
        return self.get_address(node) == 0

    def is_negation_node(self,node):
        """
        @param node: A node of a L{nltk.parse.dependencygrpah.DependencyGraph}
        @return True if it's a negation node, False otherwise.
        """
        word = self.get_word(node).lower()
        rel = self.get_rel(node)
         
        if (word == 'no' and rel in ['mod','neg']) or (word == 'nunca' and rel != 'S') or (word == 'sin'):
                return True
        return False


    def is_intensifier(self,node,dictionary):
        """
        @param node: A node of a L{nltk.parse.dependencygraph.DependencyGraph}
        @param dictionary: An instance of L{Dictionary}
        @return: True if word is an intensifier, false otherwise
        """               
        intensifier_rel = ['spec','espec','cc','sadv','f']
        intensifier_categ = ['r','f']
        lexical_category = self.get_lexical_category(node)
        lemma = dictionary.get_lemma(lexical_category, self.get_word(node))
        return (self.get_rel(node) in intensifier_rel 
                and self.get_lexical_category(node) in intensifier_categ
                and dictionary.is_intensifier_term(lemma))    

    def is_emoticon(self,node):
        """
        @param node: A node of a L{nltk.parse.dependencygraph.DependencyGraph}
        @return True if the node is an emoticon, False otherwise
        """
        return self.get_rel(node) == 'art_rel_emoticon'


    def is_artificial_node(self,node):
        """
        @param node: A node of a L{nltk.parse.dependencygraph.DependencyGraph}
        @return: True if node was created artificially by L{src.model.parser.Parser}, False otherwise
        """
        return self.get_rel(node) == 'art_rel_adversative'
    

#    def get_subgraph(self,new_root_node):
#        difference = self.get_address(new_root_node)
#        
#        subgraph = SentimentDependencyGraph()
#        return SentimentDependencyGraph()
    
    def _nodes_in_graph(self,node):    

        list_nodes = []
        if self.is_leaf(node):
            return [node]
        else:
            if not self.is_root_node(node):
                list_nodes = [node]
            list_children_node = [self.get_by_address(address) 
                                  for address in self.get_deps(node)]
            for child_node in list_children_node:
                list_nodes.extend(self._nodes_in_graph(child_node))
        return list_nodes
    
    
    def graph_to_string(self,node):
        """
        @param A node of a L{SentimentDependencyGraph}
        @return A string. The raw phrase which starts in the node. 
        """
        sorted_nodes = sorted(self._nodes_in_graph(node), key= lambda d: d['address'],reverse=False);
        return ' '.join([self.get_word(node) for node in sorted_nodes])


    def number_of_nodes(self, node):
        """
        @param node: A node of a L{SentimentDependencyGraph}
        @return An integer. The number of the nodes starting in node
        """
        number_nodes = 0;
        if self.is_leaf(node):
            return 1
        else:
            if not self.is_root_node(node):
                number_nodes = 1
            list_children_node = [self.get_by_address(address) 
                                  for address in self.get_deps(node)]
            for child_node in list_children_node:
                number_nodes+=self.number_of_nodes(child_node)
        return number_nodes

    def level(self,address,level=1):
        """
        @param dg: A L{nltk.parse.dependencygraph.DependencyGraph} instance
        @param address: An integer representing the identifier of a node of dg
        @param level: Initial level of the node, before recursive calls of level function
        @return:The level of a node in a dependency graph
        """
        try:
            head_node = self.get_by_address(self._hd(address))
        except:
            return level
        if self.is_root_node(head_node):
            return level
        else:
            return self.level(self.get_address(head_node), level+1)



    def dg_to_json(self, node_address, dictionary):
        """
        @param node_address: An integer. The identifier of a node of a L{SentimentDependencyGraph}
        @param dictionary: An instance of L{Dictionary}
        """
          
        node = self.get_by_address(node_address)
        sucessors = []
        node_dict = {SentimentJSONKeys.ADDRESS: self.get_address(node),
                     SentimentJSONKeys.WORD:self.get_word(node),
                     SentimentJSONKeys.POSTAG: self.get_tag(node),
                     SentimentJSONKeys.DEPENDENCY_TYPE: self.get_rel(node),
                     SentimentJSONKeys.CHILDREN: sucessors,
                     SentimentJSONKeys.IS_NEGATION: self.is_negation_node(node),
                     SentimentJSONKeys.IS_INTENSIFIER: self.is_intensifier(node,dictionary),
                     SentimentJSONKeys.SEMANTIC_ORIENTATION: node[SentimentDependencyGraphNodeKeys.SEMANTIC_ORIENTATION] if
                                                             SentimentDependencyGraphNodeKeys.SEMANTIC_ORIENTATION in node else None}
          
        if self.is_leaf(node):
            return node_dict
        else:
            children = self.get_deps(node)
            for child in children:
                sucessors.append(self.dg_to_json(child,dictionary))
            node_dict[SentimentJSONKeys.CHILDREN] = sucessors
            return node_dict
        
