#-*- coding: utf-8 -*- 
'''
@author: David Vilares
'''
import codecs
import os
from miopia.analyzer.Analyzer import Analyzer
from miopia.util.exceptions.UnknownSOException import UnknownSOException
from miopia.analyzer.SentimentInfo import SentimentInfo
from miopia.analyzer.NegationInfo import NegationInfo
from miopia.analyzer.NegationRules import NegationRules
from miopia.analyzer.SemanticCategory import SemanticCategory
from miopia.parser.SentimentDependencyGraph import SentimentDependencyGraphNodeKeys
from miopia.preparator.LexicalSentimentInfo import LexicalValenceShifter

class SentimentAnalyzer(Analyzer):    
    """
    Instances of L{SentimentAnalyzer} evaluates 
    L{SentimentDependencyGraph}'s of a text, returning a summary in a L{SentimentInfo} object
    """
    #TODO: Proof number_intensifiers, length_text and number_words
       
    def __init__(self, parser, dictionaries,conf,
                 preprocessor, lexical_processor):
        """
        @param parser: An instance of L{Parser}
        @param dictionaries: An instance of L{Dictionary}
        @param conf: An instance of L{AnalyzerConfiguration}
        @param preprocessor: An instance of L{PreProcessor}
        @param lexical_processor: An instance of L{LexicalProcessor}
        """
        self._conf = conf
        self._sentence_lsi = None #Auxiliar structure for lexical sentiment info
        
        self._visit_functions_dict = {SemanticCategory.NOUN: self._visit_noun,
                                      SemanticCategory.ADJECTIVE: self._visit_adjective,
                                      SemanticCategory.ADVERB: self._visit_adverb,
                                      SemanticCategory.VERB: self._visit_verb,
                                      SemanticCategory.INTENSIFIER: self._visit_intensifier,
                                      SemanticCategory.SUBORDINATE_ADVERSATIVE: self._visit_adversative,
                                      SemanticCategory.NEGATION: self._visit_negation,
                                      SemanticCategory.NEGATION_WITHOUT: self._visit_without_negation,
                                      SemanticCategory.EMOTICON: self._visit_emoticon,
                                      SemanticCategory.OTHER: self._visit_other
                                      }
        super(SentimentAnalyzer,self).__init__(parser,dictionaries,preprocessor,lexical_processor)


    def add_sentiment_info_to_node(self,node,sentiment_info):
        """
        @param node: A node of a L{SentimentDependencyGraph}
        @param sentiment_info: An instance of L{SentimentInfo} containing
        the new information for the node.
        """
        node[SentimentDependencyGraphNodeKeys.SEMANTIC_ORIENTATION] = sentiment_info.get_so()
        node[SentimentDependencyGraphNodeKeys.INTENSIFICATION] = sentiment_info.get_int()
        node[SentimentDependencyGraphNodeKeys.SUBJECTIVITY] = sentiment_info.get_subjectivity()
        node[SentimentDependencyGraphNodeKeys.POSITIVE_WORDS] = sentiment_info.get_pos_words()
        node[SentimentDependencyGraphNodeKeys.NEGATIVE_WORDS] = sentiment_info.get_neg_words()
        node[SentimentDependencyGraphNodeKeys.NUMBER_OF_INTENSIFIERS] = sentiment_info.get_number_intensifiers()
        node[SentimentDependencyGraphNodeKeys.LENGTH_TEXT] = sentiment_info.get_length_text()
        node[SentimentDependencyGraphNodeKeys.NUMBER_OF_WORDS] = sentiment_info.get_number_words()

        

    def _analyze_graphs(self, dependency_graphs, lsi):
        """
        @param dependency_graphs: A list of L{SentimentDependencyGraph}
        @return: A tuple ([L{SentimentDependencyGraph}],L{SentimentInfo}) 
        The first element is a list of the dependency graphs of the texts. Each
        node contains their own L{src.miope.anayzer.SentimentInfo}. The second element is
        SentimentInfo object containing the general information for the whole text
        """
        semantic_orientation, positive_words, negative_words = 0,0,0
        number_intensifiers, length_text, number_words = 0,0,0
        subjectivity = False
        i = 0 
        for dg in dependency_graphs:
            
            if lsi is None : self._sentence_lsi = None
            try: 
                self._sentence_lsi = lsi.get_sentence_info(i+1)
            except AttributeError: 
                self._sentence_lsi = None    
            root_node = dg.get_by_address(0) 
            self.evaluate(dg,root_node)
            subjectivity = subjectivity or root_node[SentimentDependencyGraphNodeKeys.SUBJECTIVITY]
            positive_words+= root_node[SentimentDependencyGraphNodeKeys.POSITIVE_WORDS]
            negative_words+= root_node[SentimentDependencyGraphNodeKeys.NEGATIVE_WORDS]
            number_intensifiers+= root_node[SentimentDependencyGraphNodeKeys.NUMBER_OF_INTENSIFIERS]
            length_text+= root_node[SentimentDependencyGraphNodeKeys.LENGTH_TEXT]
            number_words+= root_node[SentimentDependencyGraphNodeKeys.NUMBER_OF_WORDS]
            if i >= len(dependency_graphs)-3: 
                semantic_orientation+= self._conf.get_final_sentences_weight()*root_node[SentimentDependencyGraphNodeKeys.SEMANTIC_ORIENTATION]
            else:
                semantic_orientation+= root_node[SentimentDependencyGraphNodeKeys.SEMANTIC_ORIENTATION]
            i = i +1

        return dependency_graphs,SentimentInfo (0.,semantic_orientation,
                                                subjectivity,positive_words,
                                                negative_words, number_intensifiers,
                                                length_text, number_words)        


    def analyze_from_conll(self,file_path,**kwargs):
        """        
        It analyzes a file in ConLL-X format
        @todo: param lsi not integrated yet.
        @param file_path: The path to the ConLL file
        @param **kwargs: Optional key 'lsi' (L{LexicalSentimentInfo})
        @return A tuple ([L{SentimentDependencyGraph}],L{SentimentInfo})
        """
        try:
            lsi=kwargs['lsi']
        except KeyError:
            lsi = None
        dependency_graphs = self._parser.parse_from_conll(file_path)
        return self._analyze_graphs(dependency_graphs, lsi)                
        
  
    def analyze_dir(self,dir_path,input_encoding='utf-8'):
        """
        It analyzes a directory of plain texts
        @param dir_path: Path to the directory of plain files to be analysed
        @param input_encoding: Encoding of the texts inside the directory
        @return A list of tuples (text_id,([L{SentimentDependencyGraph}],L{SentimentInfo}))
        """        
        list_of_tagged_sentences = []
        list_id_and_linguistic_info = []
        dict_lsi = {}
        files = os.listdir(dir_path)
        for file_id in files:
            text = codecs.open(dir_path+os.sep+file_id,encoding=input_encoding).read()
            tagged_sentences, lsi = self._preanalyze(text)
            list_of_tagged_sentences.append((file_id,tagged_sentences))
            dict_lsi[file_id] = lsi
        
        files_parsed = self._parser.parse_dir(list_of_tagged_sentences)    
        for file_id,dependency_graphs in files_parsed:
            list_id_and_linguistic_info.append((file_id,
                                                self._analyze_graphs(dependency_graphs, 
                                                                    dict_lsi[file_id])[1]))
        return list_id_and_linguistic_info
 
 
    def analyze_from_conll_list(self,list_conll_files):
        raise NotImplementedError      
      
    def analyze(self, text):
        """
        @param text: The string to be analysed.. Use unicode
        @return A tuple ([L{SentimentDependencyGraph}],L{SentimentInfo})
        """     
        tagged_sentences, lsi = self._preanalyze(text)
        dependency_graphs = self._parser.parse(tagged_sentences)   
        return self._analyze_graphs(dependency_graphs, lsi)  
          

    def analyze_from_plain_file(self, file_path,input_encoding='utf-8',**kwargs):
        """
        It analyzes a plain file
        @param file_path: The path to the input plain file to be analysed
        @param input_encoding: The encoding of the file
        @return A tuple ([L{SentimentDependencyGraph}],L{SentimentInfo})
        """     
        f = codecs.open(file_path,encoding=input_encoding)
        text = f.read()
        tagged_sentences, lsi = self._preanalyze(text)
        dependency_graphs = self._parser.parse(tagged_sentences)   
        f.close()
        return self._analyze_graphs(dependency_graphs, lsi)

        


    def evaluate(self,graph,node):
        """
        This method analyzes recursively an instance of a L{SentimentDependencyGraph}
        @param graph: A L{SentimentDependencyGraph} object 
        @param node: A node of a L{SentimentDependencyGraph}
        @return: A L{SentimentInfo}
        """    
        if graph.is_leaf(node):
            sentiment_info = self.get_sentiment_info(graph,node)
            self.add_sentiment_info_to_node(node, sentiment_info)
            return sentiment_info
        else:
            semantic_category = self.get_semantic_category(graph,node)
            f = self.visit_function(semantic_category)
            return f(graph,node)
        

    
    def _get_lexical_ponderation(self,list_of_lvs):
        """
        @param list_of_lvs: A list of L{LexicalValenceShifters}
        @return: A float with the lexical weighting
        """
        ponderation = 0.
        switch = {LexicalValenceShifter.CAPS: self._conf.get_caps_int(),
                  LexicalValenceShifter.REPLICATION: self._conf.get_replication_int()}
        for lvs in list_of_lvs:
            ponderation+=switch[lvs]
        return ponderation
             
             
#    def _nesting_word_ponderation(self,graph,node):
#        """
#        @param graph: 
#        @param node: A node of the graph
#        @return A float. It represents a weighting value with respect
#        to the level of node in the graph. 
#        """
#        if not graph.is_root_node(node):
#            return (1. / graph.level(graph, graph.get_address(node)))
#        return 0

    def get_sentiment_info(self,graph,node):
        """
        @param graph: A L{SentimentDependencyGraph} object 
        @param node: A node of the graph
        @return: A instance of the L{SentimentInfo} with the sentiment info
        of that node, considered in an isolated way.
        """    
        lexical_category = graph.get_lexical_category(node)
        semantic_category = self.get_semantic_category(graph, node)
        word = graph.get_word(node)
        len_word = (lambda node: len(word) if not graph.is_root_node(node) else 0)(node)
        number_words = (lambda node: 1 if not graph.is_root_node(node) else 0)(node)
        lemma = self._dictionaries.get_lemma(lexical_category,word)
#        word_ponderation = self._nesting_word_ponderation(graph,node)
        pond = 1
        semantic_orientation = 0

        #Obtain the lexical sentiment info of the token
        if self._sentence_lsi is not None:
            try: 
                pond+=self._get_lexical_ponderation(self._sentence_lsi[graph.get_address(node)])
            except KeyError: 
                pass    
        
        if graph.is_intensifier(node,self._dictionaries):
            try:
                #XXX: With SFU was used True as subjectivity, but actually is False because we its significance has changed
                return SentimentInfo(pond*self._dictionaries.get_semantic_orientation(lemma,semantic_category),
                                     semantic_orientation, False,0,0,1,len_word,number_words)
            except UnknownSOException:
                return SentimentInfo(0,semantic_orientation, False,0,0,0,len_word,number_words)
        else:
            try:           
                pond+= 0.2 #Optimized to the SFU Corpus
                semantic_orientation = self._dictionaries.get_semantic_orientation(lemma,semantic_category)*pond   
                return SentimentInfo(0, semantic_orientation ,True,
                                     int(semantic_orientation>0),#*(1+word_ponderation),
                                     int(semantic_orientation<0),#*(1+word_ponderation),
                                     0,len_word,number_words)
            except UnknownSOException:
                return SentimentInfo(0,semantic_orientation,False,0,0,0,len_word,number_words)
                
    def visit_function(self, semantic_category):
        """
        @param semantic_category: A value of L{SemanticCategory}
        @return: An specific visit function according the semantic category of the word
        """        
        #TODO: Implement visit interjections, visit negation "ni" and visit irrealis
        return self._visit_functions_dict[semantic_category]

    
    def _compute_children(self,graph,children):
        """
        It aggregates the L{SentimentInfo} of a list of children nodes.
        @param graph: An instance of a L{SentimentDependencyGraph}
        @param children: A list of id's of the node children. Id's are integers.
        @return: A L{SentimentInfo} object with the aggregated 
        sentiment info for the children nodes
        """
        semantic_orientation_children, intensification = 0,0
        positive_words, negative_words = 0,0
        number_intensifiers, length_text, number_words =0,0,0
        subjectivity = False
         
        for child in children:
            child_node = graph.get_by_address(child)
            self.evaluate(graph,child_node)  
            positive_words+= child_node[SentimentDependencyGraphNodeKeys.POSITIVE_WORDS]
            negative_words+= child_node[SentimentDependencyGraphNodeKeys.NEGATIVE_WORDS]
            number_intensifiers+= child_node[SentimentDependencyGraphNodeKeys.NUMBER_OF_INTENSIFIERS]
            length_text+= child_node[SentimentDependencyGraphNodeKeys.LENGTH_TEXT]
            subjectivity = subjectivity or child_node[SentimentDependencyGraphNodeKeys.SUBJECTIVITY]
            number_words+= child_node[SentimentDependencyGraphNodeKeys.NUMBER_OF_WORDS]            
            intensification+= child_node[SentimentDependencyGraphNodeKeys.INTENSIFICATION]
            semantic_orientation_children+= child_node[SentimentDependencyGraphNodeKeys.SEMANTIC_ORIENTATION]
   
        return SentimentInfo(intensification, semantic_orientation_children,
                              subjectivity, positive_words,negative_words,
                              number_intensifiers,length_text,number_words)        
 
 
 
    def _default_join_semantic_orientation(self,parent, children):
        """
        The default function for joining the semantic orientation of two
        L{src.miope.analyzer.SentimentInfo} objects.
        @param parent: The L{SentimentInfo} object which contains the semantic info information of a parent node.
        @param children: The L{SentimentInfo} object which contains the semantic info information of their children.
        """
        
        return  (1+parent.get_int() + children.get_int())*parent.get_so() + children.get_so() 
    
    def _default_join_intensification(self,parent,children):
        """
        The default function for joining the semantic orientation of two
        L{src.miope.analyzer.SentimentInfo} objects.
        @param parent: The L{SentimentInfo} object which contains the semantic info information of a parent node.
        @param children: The L{SentimentInfo} object which contains the semantic info information of their children.
        """
        return parent.get_int() + children.get_int()
    
    def _swap_pos_neg_words(self,sentiment_info):
        """
        It swaps the positive and negative words of a 
        L{SentimentInfo} object.
        @param sentiment_info: The L{SentimentInfo} object
        @return: The swapped L{SentimentInfo} object
        """
        aux_pos = sentiment_info.get_pos_words()
        aux_neg = sentiment_info.get_neg_words()
        sentiment_info.set_neg_words(aux_pos)
        sentiment_info.set_pos_words(aux_neg)
        return sentiment_info
    
    
    def _join_sentiment_info(self,parent, children, 
                             f_join_semantic_orientation,
                             f_join_intensification = None):
        """
        @param parent: The parent L{SentimentInfo} object
        @param children: The joined L{SentimentInfo} object of the children
        @param f_join_semantic_orientation: A function detailing how to join
        the semantic orientation of the parent and their children
        """ 
        #intensification =  parent.get_int() + children.get_int()
        if f_join_intensification is None:
            intensification = 0.
        else:
            intensification = f_join_intensification(parent,children)
        semantic_orientation = f_join_semantic_orientation(parent, children)
        subjectivity = parent.get_subjectivity() or children.get_subjectivity()
        positive_words = parent.get_pos_words() + children.get_pos_words()
        negative_words = parent.get_neg_words() + children.get_neg_words()
        number_intensifiers = (parent.get_number_intensifiers() 
                                + children.get_number_intensifiers())
        length_text = parent.get_length_text() + children.get_length_text() 
        number_words = parent.get_number_words() + children.get_number_words()
        return SentimentInfo(intensification, semantic_orientation, 
                             subjectivity, positive_words, negative_words,
                             number_intensifiers, length_text, number_words)


    def _visit_noun(self,graph,node):
        """
        This function processes a noun node and realizes
        the recursive calls to their children.
        @param graph: L{SentimentDependencyGraph}
        @param node: A noun node of the graph
        @return A L{SentimentInfo}. It contains
        the joined sentiment info of the noun node and their children.
        """     
        node_sentiment_info = self.get_sentiment_info(graph,node) 
        children = graph.get_deps(node) 
        children_sentiment_info = self._compute_children(graph, children)
        sentiment_info = self._join_sentiment_info(node_sentiment_info, 
                                                   children_sentiment_info,
                                                   self._default_join_semantic_orientation)
        self.add_sentiment_info_to_node(node, sentiment_info)
        return sentiment_info        


    def _visit_adjective(self,graph,node):
        """
        This function processes an adjective node and realizes
        the recursive calls to their children.
        @param graph: L{SentimentDependencyGraph}
        @param node: A adjective node of the graph
        @return A L{SentimentInfo}. It contains
        the joined sentiment info of the adjective node and their children.
        """     
        node_sentiment_info = self.get_sentiment_info(graph,node) 
        children = graph.get_deps(node) 
        children_sentiment_info = self._compute_children(graph, children)
        sentiment_info = self._join_sentiment_info(node_sentiment_info, 
                                                   children_sentiment_info,
                                                   self._default_join_semantic_orientation)
        self.add_sentiment_info_to_node(node, sentiment_info)
        return sentiment_info     

    
    def _visit_verb(self,graph,node):
        """
        This function processes a verb node and realizes
        the recursive calls to their children.
        @param graph: L{SentimentDependencyGraph}
        @param node: A verb node of the graph
        @return A L{SentimentInfo}. It contains
        the joined sentiment info of the verb node and their children.
        """     
        node_sentiment_info = self.get_sentiment_info(graph,node) 
        children = graph.get_deps(node) 
        children_sentiment_info = self._compute_children(graph, children)
        sentiment_info = self._join_sentiment_info(node_sentiment_info, 
                                                   children_sentiment_info,
                                                   self._default_join_semantic_orientation)
        self.add_sentiment_info_to_node(node, sentiment_info)
        return sentiment_info     

    def _visit_adverb(self,graph,node):
        """
        This function processes an adverb node and realizes
        the recursive calls to their children.
        @param graph: L{SentimentDependencyGraph}
        @param node: An adverb node of the graph
        @return A L{SentimentInfo}. It contains
        the joined sentiment info of the adverb node and their children.
        """     
        node_sentiment_info = self.get_sentiment_info(graph,node) 
        children = graph.get_deps(node) 
        children_sentiment_info = self._compute_children(graph, children)
        sentiment_info = self._join_sentiment_info(node_sentiment_info, 
                                                   children_sentiment_info,
                                                   self._default_join_semantic_orientation)

        self.add_sentiment_info_to_node(node, sentiment_info)
        return sentiment_info     
        

    def _visit_intensifier(self,graph,node):
        """
        This function processes an intensfier node and realizes
        the recursive calls to their children.
        @param graph: L{SentimentDependencyGraph}
        @param node: An intensifier node of the graph
        @return A L{SentimentInfo}. It contains
        the joined sentiment info of the intensifier node and their children.
        """     
        node_sentiment_info = self.get_sentiment_info(graph,node) 
        children = graph.get_deps(node) 
        children_sentiment_info = self._compute_children(graph, children)
        sentiment_info = self._join_sentiment_info(node_sentiment_info, 
                                                   children_sentiment_info,
                                                   self._default_join_semantic_orientation,
                                                   self._default_join_intensification)
        self.add_sentiment_info_to_node(node, sentiment_info)
        return sentiment_info        


    
    def _visit_adversative(self,graph,node):
        """
        This function processes an artificial adversative node and realizes
        the recursive calls to their children.
        @param graph: L{SentimentDependencyGraph}
        @param node: An artificial adversative node of the graph
        @return A L{SentimentInfo}. It contains
        the joined sentiment info of the adversative node and their children.
        """     

        nodes_before_adversative = []
        nodes_after_adversative = []
           
           
           
        def is_restrict_adversative(tag):
            """
            It determines if the artificial adversative node is
            restrictive
            """
            return tag.split(":")[1].split("@")[0] == 'restrict'
           
           
        children = graph.get_deps(node)
        tag = graph.get_tag(node)
        adversative_id = graph._get_adversative_id(node)
        for c in children:
            if c < adversative_id: nodes_before_adversative.append(c)
            else: nodes_after_adversative.append(c)
        nodes_before_sentiment_info = self._compute_children(graph, 
                                                             nodes_before_adversative)
        nodes_after_sentiment_info = self._compute_children(graph, nodes_after_adversative)
               
        if is_restrict_adversative(tag):
            adv_pond = self._conf.get_restrictive_adv_weight()
            main_pond = self._conf.get_restrictive_main_weight()
        else:
            adv_pond = 0.
            main_pond = 1.
            
        join_function = lambda b,a : adv_pond*b.get_so()+main_pond*a.get_so()          
        sentiment_info = self._join_sentiment_info(nodes_before_sentiment_info, 
                                      nodes_after_sentiment_info, 
                                      join_function,
                                      self._default_join_intensification)
        self.add_sentiment_info_to_node(node, sentiment_info)
        return sentiment_info



    def _shift_negation_without(self,semantic_orientation):
        """
        A function to shift the semantic orientation of a 'without' semantic dependent words
        @param semantic_orientation: An integer
        @return The shifted semantic orientation
        """
        return (semantic_orientation + self._conf.get_without_shift() 
                * (-1 if semantic_orientation >= 0 else 1))
    
    def _shift_negation(self,semantic_orientation):
        """
        A function to shift the semantic orientation of a negation semantic dependent words
        @param semantic_orientation: An integer
        @return The shifted semantic orientation
        """
        return (semantic_orientation + self._conf.get_neg_shift() 
                * (-1 if semantic_orientation >= 0 else 1))           

    def _is_adjunct_node(self,graph,node):
        """
        It determines if the node is an adjunct branch
        @param node: A node of L{SentimentDependencyGraph}
        @return: A boolean.
        """
        list_adjunct = ['cc','creg']
        return graph.get_rel(node) in list_adjunct 

    def is_subjective_node(self,graph,node):
        """
        It determines if a node is subjective
        @param graph: An instance of L{SentimentDependencyGraph}
        @param node: A node of the graph
        @return: A boolean
        """
        return self.get_sentiment_info(graph,node).get_subjectivity()
    
    def _negated_and_other_branches(self,graph,children_nodes,inclusion_function,
                                    include_all_branches):
        """
        @param graph: An instance of L{SentimentDependencyGraph}
        @param children_nodes: A list of the children_nodes 
        @param inclusion_function: A function to determine if a node should
        be negated or not.
        @param include_all_branches: A boolean. True: It negates all the nodes
        that match the inclusion function. False: It only includes the 
        first node that match the inclusion function
        @return A tuple ([negated_nodes],[non_negated_nodes])
        """
        negated_branches = []
        other_branches = []
        include_branch = True
        for child in children_nodes:
            if inclusion_function(graph,child) and include_branch:
                negated_branches.append(graph.get_address(child))
                include_branch = include_all_branches
            else:
                other_branches.append(graph.get_address(child))    
        return negated_branches, other_branches
    
    

    def _visit_without_negation(self,graph,node):
        """
        This function processes a 'without' node and realizes
        the recursive calls to their children.
        @param graph: An instance of L{SentimentDependencyGraph}
        @param node: An artificial without node of the graph
        @return A L{SentimentInfo}. It contains
        the joined sentiment info of the without node and their children.
        """     

        node_sentiment_info =  self.get_sentiment_info(graph,node) #SentimentInfo object
        children = graph.get_deps(node)
        children_sentiment_info = self._compute_children(graph, children)
        children_sentiment_info = self._swap_pos_neg_words(children_sentiment_info)
        join_function_semantic_orientation =  lambda p,c: self._shift_negation_without(c.get_so())*(p.get_subjectivity()*
                                                                 c.get_subjectivity())
        sentiment_info = self._join_sentiment_info(node_sentiment_info, 
                                                   children_sentiment_info, 
                                                   join_function_semantic_orientation,
                                                   self._default_join_intensification)
        self.add_sentiment_info_to_node(node, sentiment_info)
        return sentiment_info

      
    def _scope_negation_detection(self,graph,node):
        """
        This method analyzes the children of a node and determines which branches must be negated and
        which not.
        @param graph: An instance of L{SentimentDependencyGraph}
        @param node: A node of the graph 
        @return: A L{NegationInfo} object
        """
        children_id = graph.get_deps(node)                           
              
        if self.is_subjective_node(graph,node): 
            return NegationInfo([],children_id, NegationRules.verb)
        children_nodes = map(graph.get_by_address,children_id)    
        #Sentimental noun rule and sentimental adjetive rule   
        neg_branch, other_branch = self._negated_and_other_branches(graph,children_nodes,
                                   lambda graph,child: graph.get_rel(child) in ['cd','atr','cpred'],
                                   True)
        if neg_branch != []:
            return NegationInfo(neg_branch,other_branch,NegationRules.branch)             
        #Sentimental cc and creg rule
        neg_branch, other_branch = self._negated_and_other_branches(graph,children_nodes,
                                   self._is_adjunct_node, False)           
        if neg_branch != []:    
                return NegationInfo(neg_branch,other_branch,NegationRules.branch)
        #Default rule  
        return NegationInfo(children_id,[],NegationRules.default)  
 

    def _negation_sentimental_word_rule(self,graph,node,**kwargs):
        """
        This function negates the parent of a negation
        @param graph: An instance of L{SentimentDependencyGraph}
        @param node: A node of the graph 
        @param **kwargs: Not used right now
        """
        node_sentiment_info = self.get_sentiment_info(graph,node) 
        children = graph.get_deps(node) 
        children_sentiment_info = self._compute_children(graph, children)
        join_function = lambda p,c: self._shift_negation((1+ p.get_int()+c.get_int())*p.get_so())*p.get_subjectivity()+ c.get_so()
        sentiment_info = self._join_sentiment_info(node_sentiment_info, 
                                                   children_sentiment_info,
                                                   join_function,
                                                   self._default_join_intensification
                                                   )
        self.add_sentiment_info_to_node(node, sentiment_info)
        return sentiment_info 
                                            
    def _negation_default_rule(self,graph,node,**kwargs):
        """
        This function negates the sibling branches of a negation
        @param graph: An instance of L{SentimentDependencyGraph}
        @param node: A node of the graph
        @param **kwargs: Not used right now 
        """
        node_sentiment_info = self.get_sentiment_info(graph,node) 
        children = graph.get_deps(node) 
        children_sentiment_info = self._compute_children(graph, children)
        children_sentiment_info = self._swap_pos_neg_words(children_sentiment_info)
        join_function = lambda p,c: self._shift_negation(p.get_so()+c.get_so())*(p.get_subjectivity() or c.get_subjectivity())
        sentiment_info = self._join_sentiment_info(node_sentiment_info, 
                                                   children_sentiment_info,
                                                   join_function,
                                                   self._default_join_intensification
                                                   )
        self.add_sentiment_info_to_node(node, sentiment_info)
        return sentiment_info         
 
    def _negation_sentimental_branch_rule(self,graph,node,**kwargs):     
        """
        This function negates the negated branches specified in kwargs['negated_branches'].
        The branches inside the parameter kwargs['other_branches'] are not negated
        @param graph: An instance of L{SentimentDependencyGraph}
        @param node: A node of the graph 
        @param **kwargs: Keys ={'negated_branches','other_branches'}.
        The parameter 'negated_branches' represent a list of nodes
        at the same level are negated. The parameter 'other_branches'
        represent a list of node at the same level that are not negated. 
        """
        node_sentiment_info = self.get_sentiment_info(graph,node)   
        negated_sentiment_info = self._compute_children(graph, 
                                                        kwargs['negated_branches'])
        negated_sentiment_info = self._swap_pos_neg_words(negated_sentiment_info)   
        other_sentiment_info = self._compute_children(graph, 
                                                      kwargs['other_branches'])
        join_function = lambda n,o: self._shift_negation(n.get_so())+o.get_so()
        joined_sentiment_info = self._join_sentiment_info(negated_sentiment_info, 
                                                   other_sentiment_info,
                                                   join_function,
                                                   self._default_join_intensification
                                                   )
        final_sentiment_info = self._join_sentiment_info(node_sentiment_info,
                                                         joined_sentiment_info, 
                                                         self._default_join_semantic_orientation,
                                                         self._default_join_intensification)
        self.add_sentiment_info_to_node(node, final_sentiment_info)
        return final_sentiment_info          
 
      

    def _negation_function(self,delimiter):
        """
        @param delimiter: L{NegationRules}
        @return: The negation function to apply given a delimiter
        """
        rules = {NegationRules.verb: self._negation_sentimental_word_rule,
                NegationRules.branch: self._negation_sentimental_branch_rule,
                NegationRules.default: self._negation_default_rule}
        return rules[delimiter]

      

    def _visit_negation(self,graph,node):
        """
        This function processes a negation node and realizes
        the recursive calls to their children.
        @param graph: L{SentimentDependencyGraph}
        @param node: A negation node of the graph
        @return A L{SentimentInfo}. It has computed
        the joined sentiment info of the negation node and their children.
        """               
        scope_of_negation = self._scope_negation_detection(graph, node)
        neg_branches = scope_of_negation.get_neg_branches()
        non_negated_branches = scope_of_negation.get_other_branches()
        type_delimiter = scope_of_negation.get_delimiter()
        return self._negation_function(type_delimiter)(graph,node,negated_branches=neg_branches,
                                            other_branches=non_negated_branches)

    def _visit_emoticon(self,graph,node):
        """
        This function processes a emoticon node and realizes
        the recursive calls to their children.
        @param graph: L{SentimentDependencyGraph}
        @param node: A emoticon node of the graph
        @return A L{SentimentInfo}. It has computed
        the joined sentiment info of the emoticon node and their children.
        """     
        node_sentiment_info = self.get_sentiment_info(graph,node) 
        children = graph.get_deps(node) 
        children_sentiment_info = self._compute_children(graph, children)
        sentiment_info = self._join_sentiment_info(node_sentiment_info, 
                                                   children_sentiment_info,
                                                   self._default_join_semantic_orientation,
                                                   self._default_join_intensification)
        self.add_sentiment_info_to_node(node, sentiment_info)
        return sentiment_info   

 
    def _visit_other(self,graph,node):
        """
        This function processes an other node and realizes
        the recursive calls to their children.
        @param graph: L{SentimentDependencyGraph}
        @param node: An other node of the graph
        @return A L{SentimentInfo}. It has computed
        the joined sentiment info of the 'other' node and their children.
        """     
        node_sentiment_info = self.get_sentiment_info(graph,node) 
        children = graph.get_deps(node) 
        children_sentiment_info = self._compute_children(graph, children)
#         if self.is_root_node(node):
#             join_function = lambda p,c: (1+p.get_int()+c.get_int())*(p.get_so()+c.get_so())
#         else:
        join_function = self._default_join_semantic_orientation
        sentiment_info = self._join_sentiment_info(node_sentiment_info, 
                                                   children_sentiment_info,
                                                   join_function,
                                                   self._default_join_intensification)
        self.add_sentiment_info_to_node(node, sentiment_info)
        return sentiment_info   

