'''
Created on 09/05/2013

@author: David Vilares
'''

class FeatureType(object):
    """
    FeatureType defines the features supported by the L{Adapter} classes
    """

    GRAM = "GRAM"
    #CROSS_POSTAG = "COARSE_POSTAG"
    #FINE_POSTAG = "FINE_POSTAG"
    DEPENDENCY = "DEPENDENCY_TYPE"
    #PSYCHOMETRIC = "PSYCHOMETRIC"
    DEPENDENCY_TRIPLET = "DEPENDENCY_TRIPLET"
    SENTIMENT_FEATURE = "SENTIMENT_FEATURE"
    LEXICON = "LEXICON"
    ABSTRACTED_LEXICON = "ABSTRACTED_LEXICON"
    SCORED = "SCORED"
    METADATA = "METADATA"
    UNGRAMMATICAL = "UNGRAMMATICAL"


class SentimentFeature(object):
    """
    SentimentFeature defines the features provided by a L{SentimentAnalyzer}
    """
    SEMANTIC_ORIENTATION = "S0"
    POS_WORDS = "POS_WORDS"
    NEG_WORDS = "NEG_WORDS"
    SUBJECTIVITY = "SUBJECTIVITY"
    NUMBER_INTENSIFIERS = "NUMBER_INTENSFIERS"
    NUMBER_WORDS = "NUMBER_WORDS"
    LENGTH_TEXT = "LENGTH_TEXT"

class FeatureSemanticProperty(object):
    """
    SemanticProperty defines a 'surname' to append semantic properties to the name of a features, building a new
    feature 
    """
    POLARITY = "POLARITY"


class FeatureLevelBackOff(object):
    """
    FeatureLevelBackOff defines back off options supported by a L{FeatureTypeConfiguration}.
    """
    FULL_BACK_OFF = "FULL_BACK_OFFF"
    NO_BACK_OFF = "NO_BACK_OFF"
    HEAD_BACK_OFF = "HEAD_BACK_OFF"
    DEPENDENT_BACK_OFF = "DEPENDENT_BACK_OFF"
    TYPE_BACK_OFF_EMPTY = "TYPE_BACK_OFF_EMPTY"
    TYPE_BACK_OFF_LEMMA = "TYPE_BACK_OFF_LEMMA"
    TYPE_BACK_OFF_WORD = "TYPE_BACK_OFF_WORD"
    TYPE_BACK_OFF_COARSE_TAG = "TYPE_BACK_OFF_COARSE_TAG"
    TYPE_BACK_OFF_FINE_TAG = "TYPE_BACK_OFF_FINE_TAG"
    TYPE_BACK_OFF_POLARITY_WORDS = "TYPE_BACK_OFF_POLARITY_WORDS"
    TYPE_BACK_OFF_SEMANTIC_ORIENTATION = "TYPE_BACK_OFF_SEMANTIC_ORIENTATION"
    TYPE_BACK_OFF_PSYCHOMETRIC = "TYPE_BACK_OFF_PSYCHOMETRIC"

    
class FeatureWekaDataType(object):
    """
    FeatureWekaDataType defines the WekaDataTypes supported by this system
    """
    NUMERIC = 'NUMERIC'


class FeatureTypeConfiguration(object):
    """
    FeatureTypeConfiguration specifies the configuration for a L{src.miope.adapter.Adapter}
    """
    DELIMITER_CONFIGURATION = "|"
    HEAD_BACK_OFF_DELIMITER = "HBOFF"
    DEPENDENT_BACK_OFF_DELIMITER = "DBOFF"
    N_GRAM_DELIMITER= "N_GRAM"
    N_GRAM_BACK_OFF_DELIMITER ="N_GRAM_BACK_OFF"
    ADD_DEPENDENCY_TYPE_DELIMITER= "DP_TYPE" 
    SEMANTIC_PROPERTY_DELIMITER = "SEMANTIC_PRO"
    DEFAULT_N_GRAM_SIZE = "1"
           
        
        
    def __init__(self, back_off_head = None, 
                 back_off_dependent = None,
                 **kwargs):
        """
        @param **kwargs: ngram, n_gram_back_off, add_dependency_type, semantic_property
        @param ngram: {"1","2","3",...} 
        @param n_gram_back_off: A type back_off of L{FeatureLevelBackOff}
        @param add_dependency_type: {True,False}. If not included, taking False by default
        @param semantic_property: A L{FeatureSemanticProperty}
        """
        
        
        self._back_off_head = back_off_head
        self._back_off_dependent = back_off_dependent
        
        if 'n_gram' in kwargs.keys():
            self._n_gram = str(kwargs['n_gram'])
        else:
            self._n_gram = None
            
        if 'n_gram_back_off' in kwargs.keys():
            self._n_gram_back_off = kwargs['n_gram_back_off']
        else:
            self._n_gram_back_off = None

        if 'add_dependency_type' in kwargs.keys():
            self._add_dependency_type = kwargs['add_dependency_type']
        else:
            self._add_dependency_type = None
            
        if 'semantic_property' in kwargs.keys():
            if kwargs['semantic_property'] in [FeatureSemanticProperty.POLARITY]:
                self._semantic_property = kwargs['semantic_property']
            else:
                raise ValueError("Not valid value for FeatureSemanticProperty")
        else:
            self._semantic_property = None
            
        if 'weka_data_type' in kwargs.keys():
            self._weka_data_type = kwargs['weka_data_type']
        else:
            self._weka_data_type = FeatureWekaDataType.NUMERIC
            
    def get_back_off_head(self):
        """
        @return: The type of back off for the head of node a dependency triplet
        """
        return self._back_off_head
    
    def get_back_off_dependent(self):
        """
        @return: The type of back off for the dependent node of a dependency triplet
        """
        return self._back_off_dependent
    
    def get_add_dependency_type(self):
        """
        @return: A boolean indicating if the L{src.miope.adapter.DependencyTripletAdapter} must include or not
        dependency types
        """
        return self._add_dependency_type
    
    def get_n_gram(self):
        """
        @return: The len for ngram's for a L{src.miope.adapter.NGramAdapter}
        """
        return self._n_gram
    
    def get_n_gram_back_off(self):
        """
        @return: The  type of back off for ngram's of a L{src.miope.adapter.NGramAdapter}
        """
        return self._n_gram_back_off
        
    
    def get_semantic_property(self):
        return self._semantic_property    
    
    def get_weka_data_type(self):
        return self._weka_data_type

    def __str__(self, *args, **kwargs):
            
        
        
        back_off_head =  ('' if 
                            self.get_back_off_head() == None 
                         else 
                            self.DELIMITER_CONFIGURATION+self.HEAD_BACK_OFF_DELIMITER+"="+self.get_back_off_head())
        back_off_dependent = ('' if 
                            self.get_back_off_dependent() == None 
                         else 
                            self.DELIMITER_CONFIGURATION+self.DEPENDENT_BACK_OFF_DELIMITER+"="+self.get_back_off_dependent())
        
    
        n_gram = ('' if 
                    self.get_n_gram() == None
                  else 
                    self.DELIMITER_CONFIGURATION+self.N_GRAM_DELIMITER+"="+self.get_n_gram())
        
        n_gram_back_off = ('' if 
                           self.get_n_gram_back_off() == None
                           else 
                           self.DELIMITER_CONFIGURATION+self.N_GRAM_BACK_OFF_DELIMITER+"="+self.get_n_gram_back_off())
        
        add_dependency_type = ('' if
                                self.get_add_dependency_type() == None
                               else
                                self.DELIMITER_CONFIGURATION+self.ADD_DEPENDENCY_TYPE_DELIMITER+"="+str(self.get_add_dependency_type()))
        
        semantic_property = ('' if
                                self.get_semantic_property() == None
                             else
                                self.DELIMITER_CONFIGURATION+self.SEMANTIC_PROPERTY_DELIMITER+"="+str(self.get_semantic_property()))
        
        return (back_off_head
                +back_off_dependent
                +n_gram
                +n_gram_back_off
                +add_dependency_type
                +semantic_property
                +self.DELIMITER_CONFIGURATION)
        
    

#    REDIFITION OF __eq__ TO COMPARE OBJECT BY THEIR ATTRIBUTES
    def __hash__(self):
        return hash((self._back_off_head, self._back_off_dependent,
                     self._n_gram, self._n_gram_back_off,
                     self._add_dependency_type, self._semantic_property))

    def __eq__(self, other):
        return (self._back_off_head, self._back_off_dependent,
                self._n_gram, self._n_gram_back_off,
                self._add_dependency_type, self._semantic_property) == (other.get_back_off_head(),
                                               other.get_back_off_dependent(),
                                               other.get_n_gram(),
                                               other.get_n_gram_back_off(),
                                               other.get_add_dependency_type(),
                                               other.get_semantic_property())
        
        
class Feature(object):
    '''
    classdocs
    '''

    def __init__(self,name,value,feature_type, type_configuration,data_type):
        '''
        Constructor
        @param name: The name of the feature.
        @param value: The number of occurences of the features.
        @param feature_type: A value of L{FeatureType}
        @param type_configuration: A L{FeatureTypeConfiguration} object
        @param data_type  A value of L{FeatureWekaDataType}
        '''
        self._name = name
        self._value = value
        self._feature_type = feature_type
        self._type_configuration = type_configuration
        self._data_type = data_type
    
    def get_name(self):
        return self._name
    
    def get_value(self):
        return self._value
        
    def get_type(self):
        return self._feature_type
    
    def get_type_configuration(self):
        return self._type_configuration
    
    def get_data_type(self):
        return self._data_type
        